import express from "express";
import cors from "cors";
import dotenv from "dotenv";
import path from "path";
import { fileURLToPath } from "url";

dotenv.config();

const app = express();
app.use(cors());
app.use(express.json({ verify: (req, _res, buf) => { req.rawBody = buf; } }));

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Serve the simple phone UI (Phase 1)
app.use("/", express.static(path.join(__dirname, "../../public")));

app.get("/health", (_req, res) => res.json({ ok: true }));

// In-memory storage for the most recent call_control_id (good enough for Phase 1 demo)
let lastCallControlId = null;

/**
 * Create an outbound call via Telnyx Call Control API
 * Requires: TELNYX_API_KEY, TELNYX_CONNECTION_ID, TELNYX_FROM_NUMBER
 */
app.post("/api/call", async (req, res) => {
  try {
    const to = (req.body?.to || "").trim();
    if (!to) return res.status(400).json({ error: "Missing 'to' number (E.164 recommended, e.g. +17135551234)" });

    const apiKey = process.env.TELNYX_API_KEY;
    const connectionId = process.env.TELNYX_CONNECTION_ID;
    const from = process.env.TELNYX_FROM_NUMBER;

    if (!apiKey) return res.status(500).json({ error: "Missing TELNYX_API_KEY in environment" });
    if (!connectionId) return res.status(500).json({ error: "Missing TELNYX_CONNECTION_ID in environment" });
    if (!from) return res.status(500).json({ error: "Missing TELNYX_FROM_NUMBER in environment" });

    const payload = { connection_id: connectionId, to, from };

    const r = await fetch("https://api.telnyx.com/v2/calls", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Authorization": `Bearer ${apiKey}`
      },
      body: JSON.stringify(payload)
    });

    const data = await r.json().catch(() => ({}));
    if (!r.ok) return res.status(r.status).json({ error: "Telnyx call create failed", details: data });

    lastCallControlId = data?.data?.call_control_id || null;

    return res.json({ ok: true, call_control_id: lastCallControlId, telnyx: data });
  } catch (err) {
    return res.status(500).json({ error: err?.message || "Unknown error" });
  }
});

/**
 * Hang up the last call (Phase 1 demo).
 */
app.post("/api/hangup", async (_req, res) => {
  try {
    const apiKey = process.env.TELNYX_API_KEY;
    if (!apiKey) return res.status(500).json({ error: "Missing TELNYX_API_KEY in environment" });

    if (!lastCallControlId) {
      return res.status(400).json({ error: "No last call_control_id stored yet. Place a call first." });
    }

    const r = await fetch(`https://api.telnyx.com/v2/calls/${lastCallControlId}/actions/hangup`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Authorization": `Bearer ${apiKey}`
      }
    });

    const data = await r.json().catch(() => ({}));
    if (!r.ok) return res.status(r.status).json({ error: "Telnyx hangup failed", details: data });

    return res.json({ ok: true, telnyx: data });
  } catch (err) {
    return res.status(500).json({ error: err?.message || "Unknown error" });
  }
});

/**
 * Telnyx webhook receiver (Phase 1: just log events)
 * Set this URL in your Telnyx Voice API Application:
 *   https://api.iquantumleads.com/webhooks/telnyx
 */
app.post("/webhooks/telnyx", (req, res) => {
  const eventType = req.body?.data?.event_type;
  console.log("TELNYX WEBHOOK EVENT:", eventType);

  // If payload contains call_control_id, store it
  const payload = req.body?.data?.payload || {};
  if (payload.call_control_id) lastCallControlId = payload.call_control_id;

  res.status(200).json({ received: true });
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`iquantumleads API running on port ${PORT}`);
});
